/*****************************************************************************
*
*  Copyright 2019 NXP
* All Rights Reserved
*
*****************************************************************************
*
* THIS SOFTWARE IS PROVIDED BY NXP "AS IS" AND ANY EXPRESSED OR
* IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
* OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
* IN NO EVENT SHALL NXP OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
* INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
* (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
* SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
* HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
* STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
* IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
* THE POSSIBILITY OF SUCH DAMAGE.
*
****************************************************************************/
// orb.cpp : Defines the entry point for the console application.

#include "descriptor.h"
#include <opencv2/opencv.hpp>

//#ifdef __STANDALONE__
#ifndef APEX2_EMULATE
  #include "f0.h"
  #include "f1.h"
#endif

#if !defined(APEX2_EMULATE) && !defined(__INTEGRITY__)
  #ifdef __STANDALONE__
    #include "frame_output_dcu.h"
  #else // #ifdef __STANDALONE__
    #include "frame_output_v234fb.h"
  #endif // else from #ifdef __STANDALONE__  
  
  #define CHNL_CNT io::IO_DATA_CH3
#endif

#include <iostream>
#include <fstream>
#include <vector>
#include <iterator>

#ifdef APEX2_EMULATE
  #include "apu_lib.hpp"
  #include "apu_extras.hpp"
  #include "acf_lib.hpp"
  #include "orbfast9_apu_process_desc.hpp"
  #include "orbmatchdescriptors_apu_process_desc.hpp"
  using namespace APEX2;
#else
  #include <stdio.h>
#include <icp_data.h>
#include <ORBFAST9.hpp>
#include <ORBMATCHDESCRIPTORS.hpp>
#include <apex.h>
  using namespace icp;
#endif

#include <umat.hpp>
#include <oal.h>

#include <string.h>
#include "common_time_measure.h"

//using namespace std;
#include "common_stringify_macros.h"

using namespace std;
using namespace cv;

#ifdef INPUT_PATH_REL
#define INPUT_ROOT __FULL_DIR__ XSTR(INPUT_PATH_REL)
#else

#ifdef APEX2_EMULATE
#define INPUT_ROOT "./A53_inc/"
#else
#define INPUT_ROOT "data/common/"
#endif

#endif
#ifdef OUTPUT_PATH_REL
#define OUTPUT_ROOT __FULL_DIR__ XSTR(OUTPUT_PATH_REL)
#else
#define OUTPUT_ROOT ""
#endif

struct Feature
{
  Point position;
  Descriptor descriptor;
  int matchedTo;
  int matchDist;

  Feature() :
    matchedTo(-1),
    matchDist(INT_MAX)
  {
  }
};

#include "orbfast9_graph_names.h"
#include "orbmatchdescriptors_graph_names.h"

int main(int argc, char* argv[])
{
  bool apu_matching = true;

#ifndef APEX2_EMULATE
  APEX_Init();
#else
  REGISTER_PROCESS_TYPE(ORBFAST9_PI, OrbFast9ApuProcessDesc)
  REGISTER_PROCESS_TYPE(ORBMATCHDESCRIPTORS_PI, OrbMatchDescriptorsApuProcessDesc)
#endif


  #include "common_time_measure.h"
  double frequency = FSL_GetFrequency();
  printf("Host frequency: %f\n", frequency);

  // Read the input using OpenCV
  #ifdef __STANDALONE__
    Mat in0(576, 768, CV_8UC1, f0);
	Mat in1(576, 768, CV_8UC1, f1);
  #else
    Mat in0 = imread(INPUT_ROOT"f0.png", 0);
    Mat in1 = imread(INPUT_ROOT"f1.png", 0);
  #endif

  if (in0.empty() || in1.empty())
  {
    printf("input images not found\n");
    return 1;
  }

  if ((in0.rows != in1.rows) || (in0.cols != in1.cols))
  {
    printf("input images have different dimensions\n");
    return 1;
  }

  int lSrcWidth = in0.cols;
  int lSrcHeight = in0.rows;
  int lRetVal = 0;

  unsigned char* tempDataF = new unsigned char[lSrcWidth * lSrcHeight];
  unsigned char* tempDataG = new unsigned char[lSrcWidth * lSrcHeight];

  // Create images
  vsdk::UMat lInput0(lSrcHeight, lSrcWidth, VSDK_CV_8UC1);
  vsdk::UMat lOutput0(lSrcHeight, lSrcWidth, VSDK_CV_8UC1);
  vsdk::UMat lOutput1( lSrcHeight, lSrcWidth, VSDK_CV_8UC1);
  vsdk::UMat lInputThreshold(1, 1, VSDK_CV_8UC1);

  lInputThreshold.getMat(ACCESS_WRITE | OAL_USAGE_CACHED).at< uint8_t >(0) = 30;

  const int reservedPoints = 1024;

  vector<KeyPoint> filteredKeyPoints0;
  vector<KeyPoint> filteredKeyPoints1;
  filteredKeyPoints0.reserve(reservedPoints);
  filteredKeyPoints1.reserve(reservedPoints);

  vector<Feature> features0;
  vector<Feature> features1;
  features0.reserve(reservedPoints);
  features1.reserve(reservedPoints);

  vector<DMatch> matches;
  matches.reserve(reservedPoints);

#ifndef __STANDALONE__
  int64 startTicks = getTickCount();
#else
  int64 startTicks = 0;
#endif

  int64 fast0Ticks = 0;
  int64 fast1Ticks = 0;
  int64 nonCopyFeatures0StartTicks = 0;
  int64 nonCopyFeatures1StartTicks = 0;
  int64 features0Ticks = 0;
  int64 features1Ticks = 0;

  double nonCopyTotalTime = .0;
  double armTime = .0;
  double apexTime = .0;

  if (!lInput0.empty() && !lOutput0.empty() && !lOutput1.empty())
  {
    // Load image into input data
	vsdk::Mat umatlInput0 = lInput0.getMat(vsdk::ACCESS_RW | OAL_USAGE_NONCACHED);

    unsigned char *inData = (unsigned char *)umatlInput0.data;
    memcpy(inData, in0.data, lSrcWidth * lSrcHeight);

    // Run the APEX process
    ORBFAST9_PI process0;

    // init
    lRetVal |= process0.Initialize();
	lRetVal |= process0.ConnectIO(GR_INPUTIMG_IN, lInput0);
	lRetVal |= process0.ConnectIO(GR_INPUTTHR_IN, lInputThreshold);
	lRetVal |= process0.ConnectIO(GR_OUTPUT0_OUT, lOutput0);
	lRetVal |= process0.ConnectIO(GR_OUTPUT1_OUT, lOutput1);

    if (lRetVal)
    {
#ifdef APEX2_EMULATE
      printf("%s\n", process0.ErrorMessage().c_str());
#endif
      delete[] tempDataF;
      delete[] tempDataG;
      return -1;
    }

    // execute for the first image
#ifndef __STANDALONE__
  int64 preFast0Ticks = getTickCount();
#else
  int64 preFast0Ticks = 0;
#endif

    lRetVal |= process0.Start();
    lRetVal |= process0.Wait();

#ifndef __STANDALONE__
  fast0Ticks = getTickCount();
#else
  fast0Ticks = 0;
#endif
    double timeElapsed = FSL_TicksToSeconds(fast0Ticks - preFast0Ticks) ;
	double timeElapsedWithCopy = FSL_TicksToSeconds(fast0Ticks - startTicks);
    printf("fast 0: %f s (%f with copying)\n", timeElapsed, timeElapsedWithCopy);

    nonCopyTotalTime += timeElapsed;
    apexTime += timeElapsed;

    vsdk::Mat umatlOutput0 = lOutput0.getMat(vsdk::ACCESS_RW | OAL_USAGE_NONCACHED);
    vsdk::Mat umatlOutput1 = lOutput1.getMat(vsdk::ACCESS_RW | OAL_USAGE_NONCACHED);

    unsigned char *outDataF = (unsigned char *)umatlOutput0.data;
    unsigned char *outDataG = (unsigned char *)umatlOutput1.data;

    const int minBorderDist = 30;

    memcpy(tempDataF, outDataF, lSrcWidth * lSrcHeight);
    memcpy(tempDataG, outDataG, lSrcWidth * lSrcHeight);

    
#ifndef __STANDALONE__
  nonCopyFeatures0StartTicks = getTickCount();
#else
  nonCopyFeatures0StartTicks = 0;
#endif

    // compute the first image's descriptors
    for (int y = minBorderDist; y < lSrcHeight - minBorderDist; ++y)
    {
      for (int x = minBorderDist; x < lSrcWidth - minBorderDist; ++x)
      {
        Point p(x, y);
        int index = y * lSrcWidth + x;
        if (tempDataF[index] > 0)
        {
          Feature f;
          float xBase, yBase;
          //GetCentroidRotation(tempDataG, lSrcWidth, p, &xBase, &yBase);
          GetCentroidRotationLUT(tempDataG, lSrcWidth, p, &xBase, &yBase);
          f.descriptor = CreateDescriptorRotated(tempDataG, lSrcWidth, p, xBase, yBase);
          f.position = p;
          features0.push_back(f);
          filteredKeyPoints0.push_back(KeyPoint(p, 1.f));
        }
      }
    }

#ifndef __STANDALONE__
  features0Ticks = getTickCount();
#else
  features0Ticks = 0;
#endif
    timeElapsed = double(features0Ticks - nonCopyFeatures0StartTicks) / frequency;
    timeElapsedWithCopy = double(features0Ticks - fast0Ticks) / frequency;
	printf("features 0: %f s (%f with copying)\n", timeElapsed, timeElapsedWithCopy);

    nonCopyTotalTime += timeElapsed;
    armTime += timeElapsed;

    memcpy(inData, in1.data, lSrcWidth * lSrcHeight);

    ORBFAST9_PI process1;
    lRetVal |= process1.Initialize();
    lRetVal |= process1.ConnectIO(GR_INPUTIMG_IN, lInput0);
	lRetVal |= process1.ConnectIO(GR_INPUTTHR_IN, lInputThreshold);
	lRetVal |= process1.ConnectIO(GR_OUTPUT0_OUT, lOutput0);
    lRetVal |= process1.ConnectIO(GR_OUTPUT1_OUT, lOutput1);
    // execute for the second image
    
#ifndef __STANDALONE__
  int64 preFast1Ticks = getTickCount();
#else
  int64 preFast1Ticks = 0;
#endif
    lRetVal |= process1.Start();
    lRetVal |= process1.Wait();

    if (lRetVal)
    {
#ifdef APEX2_EMULATE
      printf("%s\n", process1.ErrorMessage().c_str());
#endif
      delete[] tempDataF;
      delete[] tempDataG;
      return -1;
    }

#ifndef __STANDALONE__
  fast1Ticks = getTickCount();
#else
  fast1Ticks = 0;
#endif

    
    timeElapsed = double(fast1Ticks - preFast1Ticks) / frequency;
    timeElapsedWithCopy = double(fast1Ticks - features0Ticks) / frequency;
	printf("fast 1: %f s (%f with copying)\n", timeElapsed, timeElapsedWithCopy);

    nonCopyTotalTime += timeElapsed;
    apexTime += timeElapsed;

    memcpy(tempDataF, outDataF, lSrcWidth * lSrcHeight);
    memcpy(tempDataG, outDataG, lSrcWidth * lSrcHeight);

#ifndef __STANDALONE__
  nonCopyFeatures1StartTicks = getTickCount();
#else
  nonCopyFeatures1StartTicks = 0;
#endif

    // compute the second image's descriptors
    for (int y = minBorderDist; y < lSrcHeight - minBorderDist; ++y)
    {
      for (int x = minBorderDist; x < lSrcWidth - minBorderDist; ++x)
      {

        Point p(x, y);
        int index = y * lSrcWidth + x;
        if (tempDataF[index] > 0)
        {
          Feature f;
          float xBase, yBase;
          //GetCentroidRotation(tempDataG, lSrcWidth, p, &xBase, &yBase);
          GetCentroidRotationLUT(tempDataG, lSrcWidth, p, &xBase, &yBase);
          f.descriptor = CreateDescriptorRotated(tempDataG, lSrcWidth, p, xBase, yBase);
          f.position = p;
          features1.push_back(f);
          filteredKeyPoints1.push_back(KeyPoint(p, 1.f));
        }
      }
    }

#ifndef __STANDALONE__
  features1Ticks = getTickCount();
#else
  features1Ticks = 0;
#endif

    timeElapsed = double(features1Ticks - nonCopyFeatures1StartTicks) / frequency;
    timeElapsedWithCopy = double(features1Ticks - fast1Ticks) / frequency;
	printf("features 1: %f s (%f with copying)\n", timeElapsed, timeElapsedWithCopy);

    nonCopyTotalTime += timeElapsed;
    armTime += timeElapsed;
  }
  else
  {
    lRetVal = 1;
  }

  const int minDistLimit = 40;
  const int rangeTest = 40;

  double apexMatchTime = 0.f;
  if (apu_matching)
  {
    const unsigned int matchingDataChunkX = 128;
    const unsigned int matchingDataChunkY = 4;
    const unsigned int cuCount = 32;

    int64 matchTicksPre, matchTicksPost;
    //TODO: different descriptors for the frames!
    vsdk::UMat matchesData0(1, 512,  VSDK_CV_16SC1);
    vsdk::UMat matchesData1(1, 512,  VSDK_CV_16SC1);
    vsdk::UMat descriptorsData0(matchingDataChunkY, matchingDataChunkX * cuCount,  VSDK_CV_8UC1);
    vsdk::UMat descriptorsData1(matchingDataChunkY, matchingDataChunkX * cuCount, VSDK_CV_8UC1);
    vsdk::UMat descriptorConfiguration(1, 4, VSDK_CV_16UC1);


    vsdk::Mat matMatchesData0 = matchesData0.getMat(vsdk::ACCESS_RW | OAL_USAGE_NONCACHED);
    int16_t* matchesDataPtr = (int16_t*)matMatchesData0.data;

    vsdk::Mat matDescriptorConfiguration = descriptorConfiguration.getMat(vsdk::ACCESS_RW | OAL_USAGE_NONCACHED);
    uint16_t* descriptorConfigurationPtr = (uint16_t*)matDescriptorConfiguration.data;

    descriptorConfigurationPtr[0] = (uint16_t)features0.size();
    descriptorConfigurationPtr[1] = (uint16_t)features1.size();
    descriptorConfigurationPtr[2] = (uint16_t)minDistLimit;
    descriptorConfigurationPtr[3] = (uint16_t)rangeTest;

    //TODO: extremely hackish...
    vsdk::Mat matDescriptorsData0 = descriptorsData0.getMat(vsdk::ACCESS_RW | OAL_USAGE_NONCACHED);
    uint8_t* descriptorsData0Ptr = (uint8_t*)matDescriptorsData0.data;



    for (unsigned int h = 0; h < matchingDataChunkY; ++h)
    {
      for (unsigned int i = 0; i < matchingDataChunkX; ++i)
      {
        unsigned int index = h * matchingDataChunkX + i;
        int rowOffset = h * matchingDataChunkX * cuCount;
        for (unsigned int j = 0; j < cuCount; ++j)
        {
          descriptorsData0Ptr[rowOffset + j * matchingDataChunkX + i] = index < features0.size() ? features0[index].descriptor.GetByte(j) : 0;
        }
      }
    }
    vsdk::Mat matDescriptorsData1 = descriptorsData1.getMat(vsdk::ACCESS_RW | OAL_USAGE_NONCACHED);
    uint8_t* descriptorsData1Ptr = (uint8_t*)matDescriptorsData1.data;

    for (unsigned int h = 0; h < matchingDataChunkY; ++h)
    {
      for (unsigned int i = 0; i < matchingDataChunkX; ++i)
      {
        unsigned int index = h * matchingDataChunkX + i;
        int rowOffset = h * matchingDataChunkX * cuCount;
        for (unsigned int j = 0; j < cuCount; ++j)
        {
          descriptorsData1Ptr[rowOffset + j * matchingDataChunkX + i] = index < features1.size() ? features1[index].descriptor.GetByte(j) : 0;
        }
      }
    }

    ORBMATCHDESCRIPTORS_PI matchingProcess;
    lRetVal |= matchingProcess.Initialize();
    lRetVal |= matchingProcess.ConnectIO(GR_INPUT0_IN, descriptorsData0);
	lRetVal |= matchingProcess.ConnectIO(GR_INPUT1_IN, descriptorsData1);
	lRetVal |= matchingProcess.ConnectIO(GR_INPUTCONFIG_IN, descriptorConfiguration);
	lRetVal |= matchingProcess.ConnectIO(GR_OUTPUT0_OUT, matchesData0);
	lRetVal |= matchingProcess.ConnectIO(GR_OUTPUT1_OUT, matchesData1);

#ifndef __STANDALONE__
  matchTicksPre = getTickCount();
#else
  matchTicksPre = 0;
#endif

	
    lRetVal |= matchingProcess.Start();
    lRetVal |= matchingProcess.Wait();
#ifndef __STANDALONE__
  matchTicksPost = getTickCount();
#else
  matchTicksPost = 0;
#endif


    if (lRetVal)
    {
#ifdef APEX2_EMULATE
      printf("%s\n", matchingProcess.ErrorMessage().c_str());
#endif
      delete[] tempDataF;
      delete[] tempDataG;
      return -1;
    }

    for (unsigned int i0 = 0; i0 < features0.size(); ++i0)
    {
      int matchedTo = (int)matchesDataPtr[i0];
      if (matchedTo >= 0)
      {
        DMatch m;
        m.imgIdx = 0;
        m.queryIdx = (int)i0;
        m.trainIdx = matchedTo;
        m.distance = float(.0f);
        matches.push_back(m);
      }
    }

    apexMatchTime = double(matchTicksPost - matchTicksPre) / frequency;
    apexTime += apexMatchTime;
  }
  else
  {
    // match the keypoints
    // TODO: now just simple and incorrect matching, should be a stable marriage problem?
    for (unsigned int i0 = 0; i0 < features0.size(); ++i0)
    {
      //for range testing
      int hamDistA = INT_MAX;
      int hamDistB = INT_MAX;
      Feature f0 = features0[i0];
      int f1Index = -1;
      for (unsigned int i1 = 0; i1 < features1.size(); ++i1)
      {
        Feature f1 = features1[i1];
        //popcount early exit didn't help
        int d = f0.descriptor.HammingDistance(f1.descriptor);
        if (f1.matchedTo >= 0)
        {
          //overwrite... what if it's not a final match after the overwrite?
          //if (d >= f1.matchDist)
          {
            continue;
          }

          features0[f1.matchedTo].matchedTo = -1;
          features1[i1].matchedTo = -1;
        }

        if (((unsigned int)d) <= minDistLimit)
        {
          if (d < hamDistA)
          {
            hamDistB = hamDistA;
            hamDistA = d;
            f1Index = i1;
          }
          else if (d < hamDistB)
          {
            hamDistB = d;
          }
        }
        else if (d < hamDistB)
        {
          hamDistB = d;
        }
      }

      if (f1Index >= 0 && (hamDistB - hamDistA > rangeTest))
      {
        features0[i0].matchedTo = f1Index;
        features1[f1Index].matchedTo = i0;
        features0[i0].matchDist = features1[f1Index].matchDist = hamDistA;
      }
    }

    //TODO: can be performed in previous cycle when not using overwrites
    for (unsigned int i0 = 0; i0 < features0.size(); ++i0)
    {
      Feature f0 = features0[i0];
      if (f0.matchedTo >= 0)
      {
        DMatch m;
        m.imgIdx = 0;
        m.queryIdx = i0;
        m.trainIdx = f0.matchedTo;
        m.distance = float(f0.matchDist);
        matches.push_back(m);
      }
    }
  }

#ifndef __STANDALONE__
  int64 endTicks = getTickCount();
#else
  int64 endTicks = 0;
#endif

  double timeElapsed = double(endTicks - features1Ticks) / frequency;
  armTime += timeElapsed;
  if (apu_matching)
  {
    printf("matching (apu): %f s\n", apexMatchTime);
    armTime -= apexMatchTime;
  }
  else
  {
    printf("matching: %f s\n", timeElapsed);
  }

  timeElapsed = double(endTicks - startTicks) / frequency;
  printf("total: %f s (%f with copying)\n", nonCopyTotalTime, timeElapsed);
  printf("arm: %f s, apex: %f s\n", armTime, apexTime);

  Mat out(in0.rows, in0.cols + in1.cols, CV_8UC3);
  
  for (int i = 0; i < filteredKeyPoints0.size(); ++i)
	circle(in0, filteredKeyPoints0[i].pt, 2, Scalar(50, 50, 50), 2);
  
  for (int i = 0; i < filteredKeyPoints1.size(); ++i)
	circle(in1, filteredKeyPoints1[i].pt, 2, Scalar(50, 50, 50), 2);
  
  for (int j = 0; j < in0.rows; ++j)
  for (int i = 0; i < in0.cols; ++i)
  {
	  out.at<Vec3b>(j, i)[0] = in0.at<unsigned char>(j, i);
	  out.at<Vec3b>(j, i)[1] = in0.at<unsigned char>(j, i);
	  out.at<Vec3b>(j, i)[2] = in0.at<unsigned char>(j, i);
	  out.at<Vec3b>(j, i+in0.cols)[0] = in1.at<unsigned char>(j, i);
	  out.at<Vec3b>(j, i+in0.cols)[1] = in1.at<unsigned char>(j, i);
	  out.at<Vec3b>(j, i+in0.cols)[2] = in1.at<unsigned char>(j, i);
  }
  
  unsigned char color = 50;
  int sw = 0;
  for( int i = 0; i < matches.size(); i++ )
  {
      int i1 = matches[i].queryIdx;
      int i2 = matches[i].trainIdx;
     
      Point kp1 = filteredKeyPoints0[i1].pt;
	  Point kp2 = filteredKeyPoints1[i2].pt;
	  kp2.x += in0.cols;
	  Scalar col;
	  if (sw % 3 == 0)
		  col = Scalar(255, color, color);
	  if (sw % 3 == 1)
		  col = Scalar(color, 255, color);
	  if (sw % 3 == 2)
		  col = Scalar(color, color, 255);
	  circle(out, kp1, 2, col, 2);
	  circle(out, kp2, 2, col, 2);
	  line( out, kp1, kp2, col, 1, CV_AA);
	  color += 50;
	  sw += 1;
	  
    
    }
 // drawMatches(in0, filteredKeyPoints0, in1, filteredKeyPoints1, matches, out);
  
  //////////////////////////////////////////////////
  // Process the output - if standalone, compare with reference
  //                    - if OS, save the images
#if !defined(APEX2_EMULATE) && !defined(__INTEGRITY__)

  // Initialize different output class for Standalone and Linux
  #ifdef __STANDALONE__
    io::FrameOutputDCU output(1280, 720,  io::IO_DATA_DEPTH_08, CHNL_CNT);
  #else
    io::FrameOutputV234Fb output(1280, 720, io::IO_DATA_DEPTH_08, CHNL_CNT);
  #endif

  // Output buffer (screen size) and it's mapped version (using cv mat in order to have copyTo functions)
  vsdk::UMat output_umat = vsdk::UMat(720, 1280, VSDK_CV_8UC3);
  {
    cv::Mat    output_mat = output_umat.getMat(ACCESS_WRITE | OAL_USAGE_CACHED);
    memset(output_mat.data, 0, 720*1280*3);

    cv::resize(out, out, cv::Size(0, 0), 1280.0 / out.cols, 1280.0 / out.cols, cv::INTER_NEAREST);
    out.copyTo(output_mat(cv::Rect(0, (720-out.rows)/2, out.cols, out.rows)));
  }

  output.PutFrame(output_umat);
#else
  std::vector<int> params;
  params.push_back(CV_IMWRITE_PNG_COMPRESSION);
  params.push_back(0);

  imwrite("out.png", out, params);
#endif


#ifdef APEX2_EMULATE
  imshow("matches", out);
  waitKey();
#endif

  return lRetVal;
}
